﻿#########################################
# SCRIPT 1 (runs normally, unchanged)
#########################################

# Originally created by github.com/cjee21.
# License: MIT.
# Modified to pull Baseboard information and simplify BIOS version for readability, also moving away from depreciated WMI query.

#Version
Write-Host "Check UEFI Secure Boot KEK and DB - Version 1.0"

# Check for admin
Write-Host "Checking for Administrator permission..."
if (-NOT ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    Write-Warning "Insufficient permissions to run this script. Please run as administrator."
    Break
} else {
    Write-Host "Running as administrator - continuing execution...`n"
}

# Print computer info
Get-Date -Format 'dd MMMM yyyy'

$computer  = Get-CimInstance Win32_ComputerSystem
$bios      = Get-CimInstance Win32_BIOS
$baseboard = Get-CimInstance Win32_BaseBoard

"Manufacturer: " + $computer.Manufacturer
"Model: " + $computer.Model

$biosinfo = $bios.SMBIOSBIOSVersion -join ", "
"BIOS: " + $biosinfo

"Baseboard: " + ($baseboard.Product)

$v = Get-ItemProperty -Path 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion'
"Windows version: {0} (Build {1}.{2})`n" -f $v.DisplayVersion, $v.CurrentBuildNumber, $v.UBR


# Check for Secure Boot status
Write-Host "Secure Boot status: " -NoNewLine
try {
    $status = Confirm-SecureBootUEFI -ErrorAction Stop
    if ($status -eq $True) {
        Write-Host "$([char]0x1b)[92mEnabled$([char]0x1b)[0m`n"
    }
    elseif ($status -eq $False) {
        Write-Host "$([char]0x1b)[91mDisabled$([char]0x1b)[0m`n"
    }
}
catch [System.PlatformNotSupportedException] {
    Write-Host "$([char]0x1b)[91mNot available$([char]0x1b)[0m`n"
    Break
}
catch {
    Write-Host "$([char]0x1b)[91mUnknown$([char]0x1b)[0m`n"
    Break
}

$bold = "$([char]0x1b)[1m"
$reset = "$([char]0x1b)[0m"
$check = "$([char]0x1b)[92m$([char]8730)$reset"
$cross =  "$([char]0x1b)[91mX$reset"

# Check whether it is ARM architecture
$IsArm = $false
try {
    $arch = (Get-CimInstance Win32_Processor -ErrorAction Stop).Architecture
    # 0 = x86, 9 = x64, 5 = ARM, 12 = ARM64
    if ($arch -eq 5 -or $arch -eq 12) {
        $IsArm = $true
        Write-Host "Detected Windows on ARM architecture!`n"
    }
} catch {
    Write-Warning "Unable to determine CPU architecture, proceeding with defaults (x64).`n"
}

function Show-UEFICertIsPresent {
    param (
        [Parameter(Mandatory)]
        [string]$SecureBootUEFIVar,
        [Parameter(Mandatory)]
        [string]$CertName
    )
    try {
        if ([System.Text.Encoding]::ASCII.GetString((Get-SecureBootUEFI $SecureBootUEFIVar -ErrorAction Stop).bytes) -match $CertName) {
            $revoked = $false
            try {
                $revoked = [System.Text.Encoding]::ASCII.GetString((Get-SecureBootUEFI dbx -ErrorAction Stop).bytes) -match $CertName
            } catch {
                $revoked = $false
            }
            Write-Host "$check $CertName (revoked: $revoked)"
        } else {
            Write-Host "$cross $CertName"
        }
    } catch {
        Write-Warning "Failed to query UEFI variable '$SecureBootUEFIVar' for cert '$CertName'"
    }
}

Write-Host $bold'Current UEFI KEK'$reset
Show-UEFICertIsPresent -SecureBootUEFIVar KEK -CertName 'Microsoft Corporation KEK CA 2011'
Show-UEFICertIsPresent -SecureBootUEFIVar KEK -CertName 'Microsoft Corporation KEK 2K CA 2023'

Write-Host ""
Write-Host $bold'Default UEFI KEK'$reset
if ($IsArm) {
    Write-Warning "Some ARM-based Windows devices can't retrieve default UEFI variables."
}
Show-UEFICertIsPresent -SecureBootUEFIVar KEKDefault -CertName 'Microsoft Corporation KEK CA 2011'
Show-UEFICertIsPresent -SecureBootUEFIVar KEKDefault -CertName 'Microsoft Corporation KEK 2K CA 2023'

Write-Host ""
Write-Host $bold'Current UEFI DB'$reset
Show-UEFICertIsPresent -SecureBootUEFIVar db -CertName 'Microsoft Windows Production PCA 2011'
Show-UEFICertIsPresent -SecureBootUEFIVar db -CertName 'Microsoft Corporation UEFI CA 2011'
Show-UEFICertIsPresent -SecureBootUEFIVar db -CertName 'Windows UEFI CA 2023'
Show-UEFICertIsPresent -SecureBootUEFIVar db -CertName 'Microsoft UEFI CA 2023'
Show-UEFICertIsPresent -SecureBootUEFIVar db -CertName 'Microsoft Option ROM UEFI CA 2023'

Write-Host ""
Write-Host $bold'Default UEFI DB'$reset
if ($IsArm) {
    Write-Warning "Some ARM-based Windows devices can't retrieve default UEFI variables."
}
Show-UEFICertIsPresent -SecureBootUEFIVar dbDefault -CertName 'Microsoft Windows Production PCA 2011'
Show-UEFICertIsPresent -SecureBootUEFIVar dbDefault -CertName 'Microsoft Corporation UEFI CA 2011'
Show-UEFICertIsPresent -SecureBootUEFIVar dbDefault -CertName 'Windows UEFI CA 2023'
Show-UEFICertIsPresent -SecureBootUEFIVar dbDefault -CertName 'Microsoft UEFI CA 2023'
Show-UEFICertIsPresent -SecureBootUEFIVar dbDefault -CertName 'Microsoft Option ROM UEFI CA 2023'


#########################################
# SCRIPT 2 (runs silently, only final log line shown)
#########################################

# Wrap Script 2 in a script block
$script2 = {
    # original content of script 2 goes here (unchanged)
    # ...
    # END of script2
}

# Capture and suppress all normal output
$CsvLog = $null
$allOutput = & {
    # region === BEGIN SCRIPT 2 ===
    $CsvLog = Join-Path $PSScriptRoot "UEFI_Certificate_Check.csv"
    function Sanitize {
        param([string]$s)
        if ($null -eq $s) { return "" }
        return ($s -replace "`r?`n", ' ') -replace '"', "''"
    }
    $today = Get-Date
    $computer  = Get-CimInstance Win32_ComputerSystem
    $bios      = Get-CimInstance Win32_BIOS
    $baseboard = Get-CimInstance Win32_BaseBoard
    $biosinfo = $bios.SMBIOSBIOSVersion -join ", "
    $v = Get-ItemProperty -Path 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion'
    $winver = "Windows version: {0} (Build {1}.{2})" -f $v.DisplayVersion, $v.CurrentBuildNumber, $v.UBR
    $status = $null
    try {
        $sb = Confirm-SecureBootUEFI -ErrorAction Stop
        if ($sb -eq $True) { $status = "Enabled" }
        elseif ($sb -eq $False) { $status = "Disabled" }
    } catch [System.PlatformNotSupportedException] { $status = "Not available" }
    catch { $status = "Unknown" }

    $IsArm = $false
    $archDesc = "Unknown"
    try {
        $arch = (Get-CimInstance Win32_Processor -ErrorAction Stop).Architecture
        switch ($arch) {
            0 { $archDesc = "x86" }
            9 { $archDesc = "x64" }
            5 { $archDesc = "ARM"; $IsArm = $true }
            12 { $archDesc = "ARM64"; $IsArm = $true }
            default { $archDesc = "Other($arch)" }
        }
    } catch { $archDesc = "Unknown" }

    function Test-UEFICert {
        param([string]$VarName,[string]$CertName)
        try {
            $bytes = (Get-SecureBootUEFI $VarName -ErrorAction Stop).Bytes
            $text = [System.Text.Encoding]::ASCII.GetString($bytes)
            $present = $false
            if ($text -match [regex]::Escape($CertName)) { $present = $true }
            if ($present) {
                $revoked = $false
                try {
                    $dbxbytes = (Get-SecureBootUEFI dbx -ErrorAction Stop).Bytes
                    $dbxtext = [System.Text.Encoding]::ASCII.GetString($dbxbytes)
                    if ($dbxtext -match [regex]::Escape($CertName)) { $revoked = $true }
                } catch { $revoked = $false }
                if ($revoked) { return "Present_revoked" } else { return "Present" }
            } else { return "Missing" }
        } catch { return "Error" }
    }

    $checks = @(
        @{ Var='KEK';       Name='Microsoft Corporation KEK CA 2011' },
        @{ Var='KEK';       Name='Microsoft Corporation KEK 2K CA 2023' },
        @{ Var='KEKDefault';Name='Microsoft Corporation KEK CA 2011' },
        @{ Var='KEKDefault';Name='Microsoft Corporation KEK 2K CA 2023' },
        @{ Var='db';        Name='Microsoft Windows Production PCA 2011' },
        @{ Var='db';        Name='Microsoft Corporation UEFI CA 2011' },
        @{ Var='db';        Name='Windows UEFI CA 2023' },
        @{ Var='db';        Name='Microsoft UEFI CA 2023' },
        @{ Var='db';        Name='Microsoft Option ROM UEFI CA 2023' },
        @{ Var='dbDefault'; Name='Microsoft Windows Production PCA 2011' },
        @{ Var='dbDefault'; Name='Microsoft Corporation UEFI CA 2011' },
        @{ Var='dbDefault'; Name='Windows UEFI CA 2023' },
        @{ Var='dbDefault'; Name='Microsoft UEFI CA 2023' },
        @{ Var='dbDefault'; Name='Microsoft Option ROM UEFI CA 2023' }
    )

    $summary = [ordered]@{}
    $summary.Timestamp       = $today.ToString('yyyy-MM-dd HH:mm:ss')
    $summary.Computer        = $env:COMPUTERNAME
    $summary.Manufacturer    = Sanitize $computer.Manufacturer
    $summary.Model           = Sanitize $computer.Model
    $summary.BIOS            = Sanitize $biosinfo
    $summary.Baseboard       = Sanitize $baseboard.Product
    $summary.WindowsVersion  = Sanitize $winver
    $summary.SecureBoot      = $status
    $summary.Architecture    = $archDesc

    foreach ($c in $checks) {
        $safeName = ($c.Var + "_" + ($c.Name -replace '[^0-9A-Za-z]', '_'))
        if ($safeName.Length -gt 80) { $safeName = $safeName.Substring(0,80) }
        $result = Test-UEFICert -VarName $c.Var -CertName $c.Name
        $summary[$safeName] = $result
    }

    $row = [PSCustomObject]$summary
    if (Test-Path $CsvLog) {
        $row | Export-Csv -Path $CsvLog -NoTypeInformation -Encoding UTF8 -Append
    } else {
        $row | Export-Csv -Path $CsvLog -NoTypeInformation -Encoding UTF8
    }
    # region === END SCRIPT 2 ===
} *>&1 | Out-Null

# Only show this one line:
Write-Host "`nCSV log updated/appended: $CsvLog"
